unit JobShopUnit;

interface

Uses
  Classes,
  Tomas;

Type
  TGenerate = class;
  TPrepare = class;
  TTransport = class;
  TExecute = class;
  TPackage = class;
  TJob = class;
  TTask = class;

  TGenerate = class(TomasElement)
    public 
    NewJob: TJob;
    NrOfTasks: Integer;
    NewTask: TTask;
    InterArrivals: TExponentialDistribution;
    TaskDurations: TUniformDistribution;
    NrOfTasksPerJob: TNormalDistribution;
    published
    Procedure Process; override;
  End;

  TPrepare = class(TomasElement)
    public
    JobList: TomasQueue;
    Job: TJob;
    published
    Procedure Process; override;
  End;

  TTransport = class(TomasElement)
    public
    JobList: TomasQueue;
    Job: TJob;
    Task: TTask;
    published
    Procedure Process; override;
  End;

  TExecute = class(TomasElement)
    public
    JobList: TomasQueue;
    Job: TJob;
    Task: TTask;
    published
    Procedure Process; override;
  End;

  TPackage = class(TomasElement)
    public 
    JobList: TomasQueue;
    Job: TJob;
    published
    Procedure Process; override;
  End;

  TJob = class(TomasElement)
    public
    TaskList: TomasQueue;
  End;

  TTask = class(TomasElement)
    public
    ReadyTime: Double;
    Duration: Double;
  End;


implementation

Var
  Generator: TGenerate;
  Preparation: TPrepare;
  Transportation: TTransport;
  Execution: TExecute;
  Packaging: TPackage;

Procedure TGenerate.Process;
Begin
  While TRUE Do
  Begin
    Hold(InterArrivals.Sample);
    NewJob:=TJob.Create('Job');
    NewJob.TaskList:=TomasQueue.Create('Tasks');
    NrOfTasks:=Round(NrOfTasksPerJob.Sample);
    While NrOfTasks > 0 Do
    Begin
      NewTask:=TTask.Create('Task');
      NewTask.Duration:=TaskDurations.Sample;
      NewJob.TaskList.AddToTail(NewTask);
      Dec(NrOfTasks);
    End;
    Preparation.JobList.AddToTail(NewJob);
    If Preparation.Status = Passive Then
      Preparation.Resume(TNow);
  End;
End;

Procedure TPrepare.Process;
Begin
  While TRUE Do
  Begin
    While JobList.Length > 0 Do
    Begin
      Job:=JobList.FirstElement;
      Hold(Job.QueueTime(JobList) + 1 - TNow);
      JobList.Remove(Job);
      Transportation.JobList.AddToTail(Job);
      If Transportation.Status = Passive Then
        Transportation.Resume(TNow);
    End;
    Suspend;
  End;
End;

Function ReadySort(J1,J2: pointer): Boolean;
Var
  Task1: TTask;
  Task2: TTask;
Begin
  Task1:=TJob(J1).TaskList.FirstElement;
  Task2:=TJob(J2).TaskList.FirstElement;
  Result:=(Task1.ReadyTime < Task2.ReadyTime);
End;

Procedure TTransport.Process;
Begin
  While TRUE Do
  Begin
    While JobList.Length > 0 Do
    Begin
      Job:=JobList.FirstElement;
      Hold(Job.QueueTime(JobList) + 0.5 - TNow);
      JobList.Remove(Job);
      If Job.TaskList.Length = 0 Then
      Begin
        Packaging.JobList.AddToTail(Job);
        If Packaging.Status = Passive Then
          Packaging.Resume(TNow);
      End
      Else
      Begin
        Task:=Job.TaskList.FirstElement;
        Task.ReadyTime:=TNow + Task.Duration;
        Execution.JobList.AddSorted(Job,ReadySort);
        If Job = Execution.JobList.FirstElement Then
        Begin
          Execution.Cancel;
          Execution.Start(TNow);
        End;
      End;
    End;
    Suspend;
  End;
End;

Procedure TExecute.Process;
Begin
  While TRUE Do
  Begin
    While JobList.Length > 0 Do
    Begin
      Job:=JobList.FirstElement;
      Task:=Job.TaskList.FirstElement;
      Hold(Task.ReadyTime - TNow);
      Job.TaskList.Remove(Task);
      Task.Free;
      JobList.Remove(Job);
      Transportation.JobList.AddTotail(Job);
      If Transportation.Status = passive Then
        Transportation.Resume(TNow);
    End;
    Suspend;
  End;
End;

Procedure TPackage.Process;
Begin
  While TRUE Do
  Begin
    While JobList.Length > 0 Do
    Begin
      Job:=JobList.FirstElement;
      Hold(Job.QueueTime(JobList) + 2 - TNow);
      JobList.Remove(Job);
      Job.TaskList.Free;
      Job.Free;
    End;
    Suspend;
  End;
End;

Procedure InitializeJobShop;
Begin
  Generator:=TGenerate.Create('Generator');
  Generator.InterArrivals:=TExponentialDistribution.Create(520623,1.0/3.0);
  Generator.TaskDurations:=TUniformDistribution.Create(122998,1,7);
  Generator.NrOfTasksPerJob:=TNormalDistribution.Create(984075,5,1);
  Generator.Start(TNow);
  Preparation:=TPrepare.Create('Preparation');
  Preparation.JobList:=TomasQueue.Create('Prep_jobs');
  Preparation.Start(TNow);
  Transportation:=TTransport.Create('Transportation');
  Transportation.JobList:=TomasQueue.Create('Transport_jobs');
  Transportation.Start(TNow);
  Execution:=TExecute.Create('Execution');
  Execution.JobList:=TomasQueue.Create('Exec_jobs');
  Execution.Start(TNow);
  Packaging:=TPackage.Create('Packaging');
  Packaging.JobList:=TomasQueue.Create('Pack_jobs');
  Packaging.Start(TNow);
End;

Initialization
  InitializeJobShop;
end.
